/*
 * Copyright (c) 2001, 2021, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.internal.reflect;

import java.lang.reflect.*;
import jdk.internal.misc.Unsafe;

/** Used only for the first few invocations of a Method; afterward,
    switches to bytecode-based implementation */

class NativeMethodAccessorImpl extends MethodAccessorImpl {
     private static final Unsafe U = Unsafe.getUnsafe();
     private static final long GENERATED_OFFSET
        = U.objectFieldOffset(NativeMethodAccessorImpl.class, "generated");

    private final Method method;
    private final DelegatingMethodAccessorImpl parent;
    private int numInvocations;
    private volatile int generated;

    NativeMethodAccessorImpl(Method method) {
        this.method = method;
        this.parent = new DelegatingMethodAccessorImpl(this);
    }

    public Object invoke(Object obj, Object[] args)
        throws IllegalArgumentException, InvocationTargetException
    {
        // We can't inflate methods belonging to hidden classes because
        // that kind of class can't be referred to by name, hence can't be
        // found from the generated bytecode.
        if (++numInvocations > ReflectionFactory.inflationThreshold()
                && !method.getDeclaringClass().isHidden()
                && generated == 0
                && U.compareAndSetInt(this, GENERATED_OFFSET, 0, 1)) {
            try {
                MethodAccessorImpl acc = (MethodAccessorImpl)
                    new MethodAccessorGenerator().
                        generateMethod(method.getDeclaringClass(),
                                       method.getName(),
                                       method.getParameterTypes(),
                                       method.getReturnType(),
                                       method.getExceptionTypes(),
                                       method.getModifiers());
                parent.setDelegate(acc);
            } catch (Throwable t) {
                // Throwable happens in generateMethod, restore generated to 0
                generated = 0;
                throw t;
            }
        }

        return invoke0(method, obj, args);
    }

    DelegatingMethodAccessorImpl getParent() {
        return parent;
    }

    private static native Object invoke0(Method m, Object obj, Object[] args);
}
